<?php
declare(strict_types=1);

// Inicialización de Sesión Segura
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Configuración y Variables
$error_mensaje = '';
$mensaje_exito = '';
$mostrar_formulario = false;
$MIN_PASSWORD_LENGTH = 10;

// Obtención y sanitización del token (Prioridad POST sobre GET)
$token_input = $_POST['token'] ?? $_GET['token'] ?? '';
$token = trim(preg_replace('/[^a-f0-9]/i', '', $token_input)); // Solo hex chars permitidos

try {
    // 1. Cargar Dependencias
    require_once __DIR__ . '/db/db_connection.php';
    
    // Cargar Branding (Manejo de error si falta el archivo)
    $brandingFile = __DIR__ . '/brand/branding.php';
    if (file_exists($brandingFile)) {
        include_once $brandingFile;
    } else {
        // Fallback básico si falla branding
        $branding = [
            'full_title' => 'Restablecer Contraseña',
            'favicon' => '',
            'login-bg' => '',
            'flow-logo' => '',
            'name' => 'CRM',
            'crm' => 'System',
            'biz_name' => 'Empresa'
        ];
    }

    // 2. Lógica GET: Verificar validez del Token al entrar al link
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        if (!empty($token)) {
            // CORRECCIÓN: Usamos 'user_id' según DataBase.sql
            $stmt = $pdo->prepare("
                SELECT user_id 
                FROM users 
                WHERE reset_token = :token 
                AND reset_token_expiry > NOW() 
                LIMIT 1
            ");
            $stmt->execute([':token' => $token]);
            $user_exists = $stmt->fetchColumn();

            if ($user_exists) {
                $mostrar_formulario = true;
            } else {
                $error_mensaje = 'El enlace no es válido o ha expirado. Por favor, solicita uno nuevo.';
            }
        } else {
            $error_mensaje = 'Acceso Denegado. No se proporcionó un token de seguridad.';
        }
    }

    // 3. Lógica POST: Procesar el cambio de contraseña
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Validación CSRF implícita por token de reset, más validación de campos
        $new_password = $_POST['password'] ?? '';
        
        if (empty($token)) {
            $error_mensaje = 'Token faltante. Reinicie el proceso.';
        } elseif (strlen($new_password) < $MIN_PASSWORD_LENGTH) {
            $error_mensaje = "La seguridad es primero. La contraseña debe tener al menos {$MIN_PASSWORD_LENGTH} caracteres.";
            $mostrar_formulario = true;
        } else {
            // Verificar token nuevamente antes de escribir (Doble Check)
            $stmt = $pdo->prepare("
                SELECT user_id 
                FROM users 
                WHERE reset_token = :token 
                AND reset_token_expiry > NOW() 
                LIMIT 1
            ");
            $stmt->execute([':token' => $token]);
            $final_user_id = $stmt->fetchColumn();

            if ($final_user_id) {
                try {
                    $pdo->beginTransaction();

                    // Hashing seguro
                    $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);

                    // CORRECCIÓN CRÍTICA: 'WHERE user_id = :id' en lugar de 'id'
                    $update_stmt = $pdo->prepare("
                        UPDATE users 
                        SET password_hash = :password, 
                            reset_token = NULL, 
                            reset_token_expiry = NULL,
                            updated_at = NOW()
                        WHERE user_id = :id
                    ");
                    
                    $update_stmt->execute([
                        ':password' => $hashed_password,
                        ':id'       => $final_user_id
                    ]);

                    $pdo->commit();
                    
                    $mensaje_exito = '¡Éxito! Tu contraseña ha sido actualizada. Ya puedes iniciar sesión.';
                    $mostrar_formulario = false; 
                    $token = ''; 
                    
                } catch (PDOException $e) {
                    if ($pdo->inTransaction()) {
                        $pdo->rollBack();
                    }
                    error_log("Error Crítico SQL en Reset Password: " . $e->getMessage());
                    $error_mensaje = 'Error interno del sistema. Inténtalo más tarde.';
                    $mostrar_formulario = true;
                }
            } else {
                $error_mensaje = 'Seguridad: El token ha expirado durante el proceso. Solicita un nuevo enlace.';
                $mostrar_formulario = false;
            }
        }
    }

} catch (PDOException $e) {
    error_log("DB Connection Error: " . $e->getMessage());
    $error_mensaje = 'Error de conexión. Contacte al administrador.';
} catch (Throwable $e) {
    error_log("General Error: " . $e->getMessage());
    $error_mensaje = 'Ocurrió un error inesperado.';
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>Restablecer Contraseña <?php echo htmlspecialchars($branding['full_title']); ?></title>
    <meta name="robots" content="noindex, nofollow">
    
    <link rel="icon" type="image/png" href="<?php echo htmlspecialchars($branding['favicon']); ?>">
    <link rel="apple-touch-icon" href="<?php echo htmlspecialchars($branding['favicon']); ?>">
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Barlow:wght@300;400;500;600;700;800;900&display=swap">

    <?php include 'files/gtm-head.php'; ?>
    
    <script src="https://unpkg.com/lucide@latest"></script>
    <link rel="stylesheet" href="style.css">
    <script src="files/header-manager.js"></script>
  
  <style>
    .fld{ position:relative; }
    .fld input{ padding-top:1.35rem; }
    .fld label{ position:absolute; left:0.875rem; top:0.95rem; font-size:.95rem; color:#6b7280; transition:all .18s ease; pointer-events:none; }
    .fld:focus-within label,
    .fld input:not(:placeholder-shown) + label{ top:.35rem; font-size:.75rem; color:var(--color-primary); }
  </style>
</head>
<body class="h-screen flex flex-col md:flex-row overflow-hidden bg-gray-50">
  
  <main class="w-full md:w-1/2 flex items-center justify-center p-6 sm:p-10 mx-auto overflow-y-auto">
    <div class="w-full max-w-md">

      <section class="md:hidden relative w-full h-48 mb-8 rounded-xl overflow-hidden shadow-md">
        <img src="<?php echo htmlspecialchars($branding['login-bg']); ?>" alt="CRM Background"
             class="absolute inset-0 w-full h-full object-cover">
        <div class="absolute inset-0 bg-gradient-to-tr from-[var(--color-primary)]/70 via-[var(--color-primary)]/40 to-[var(--color-secondary)]/40"></div>
        <div class="relative z-10 h-full flex items-center justify-center">
              <img src="<?php echo htmlspecialchars($branding['flow-logo']); ?>" alt="<?php echo htmlspecialchars($branding['full_title']); ?>" class="h-16 w-auto drop-shadow-lg">
        </div>
      </section>
      
      <header class="mb-6 text-center">
        <h2 class="text-4xl md:text-5xl font-black leading-none uppercase">
          <span class="text-[var(--color-primary)]">NUEVA</span>
          <span class="text-[var(--color-secondary)]">CLAVE</span>
        </h2>
        <p class="mt-3 text-sm font-semibold uppercase text-gray-500">
          Establece Una Contraseña Segura Para Tu Cuenta <span class="text-[var(--color-primary)]">
                    <?php echo htmlspecialchars($branding['name']); ?><span class="text-[var(--color-secondary)] font-black"><?php echo htmlspecialchars($branding['crm']); ?></span>™
                </span>
        </p>
      </header>

      <?php if (!empty($error_mensaje)): ?>
        <div class="mb-4 p-3 rounded bg-red-100 text-red-700 border-l-4 border-[var(--color-secondary)] flex items-center gap-2">
          <i data-lucide="alert-triangle" class="w-6 h-6 text-[var(--color-secondary)] shrink-0"></i>
          <span class="font-bold text-md uppercase"><?php echo htmlspecialchars($error_mensaje); ?></span>
        </div>
      <?php endif; ?>

      <?php if (!empty($mensaje_exito)): ?>
        <div class="mb-4 p-3 rounded bg-green-100 text-green-700 border-l-4 border-green-500 flex items-center gap-2">
          <i data-lucide="check-circle" class="w-5 h-5"></i>
          <span class="font-bold text-sm"><?php echo htmlspecialchars($mensaje_exito); ?></span>
        </div>
      <?php endif; ?>

      <?php if ($mostrar_formulario): ?>
      <form action="/reset_password.php" method="POST" class="space-y-5">
        
        <div class="fld relative">
          <input
            type="password" id="password" name="password" placeholder=" " required minlength="<?php echo $MIN_PASSWORD_LENGTH; ?>" autocomplete="new-password"
            class="w-full peer rounded-xl border border-gray-300 bg-white/90 px-3.5 pt-5 pb-2.5 text-[15px] leading-tight outline-none transition focus:border-[var(--color-secondary)] focus:ring-2 focus:ring-[var(--color-highlight)]/50"
          />
          <label for="password" class="absolute left-3.5 top-0.5 text-xs text-gray-500 transition-all peer-placeholder-shown:top-4 peer-placeholder-shown:text-base peer-focus:top-0.5 peer-focus:text-xs">
            Nueva Contraseña (Mín. <?php echo $MIN_PASSWORD_LENGTH; ?> Caracteres)
          </label>
          <button type="button" id="togglePass" aria-label="Mostrar u ocultar contraseña"
                  class="absolute right-3 top-3.5 p-1 rounded-md text-gray-500 hover:text-gray-700 focus:outline-none">
            <i data-lucide="eye" class="w-5 h-5"></i>
          </button>
        </div>

        <input type="hidden" name="token" value="<?php echo htmlspecialchars($token); ?>">

        <button type="submit"
                class="w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 shadow-lg transition-transform transform hover:-translate-y-0.5">
          <i data-lucide="key-round" class="w-5 h-5"></i>
          <span>Establecer Nueva Contraseña</span>
        </button>
      </form>
      <?php endif; ?>

      <p class="mt-6 text-center text-gray-600">
        <a href="login.php"
           class="font-black uppercase text-sm text-[var(--color-primary)] hover:text-[var(--color-secondary)] transition-colors inline-flex items-center gap-2">
           <i data-lucide="log-in" class="w-4 h-4"></i> Volver Al Inicio De Sesión
        </a>
      </p>

      <div class="mt-12 text-center">
        <div class="max-w-7xl mx-auto px-6 lg:px-10 text-center">
            <p class="text-base font-semibold text-black">
                © <?php echo date('Y'); ?> <?php echo htmlspecialchars($branding['biz_name'] ?? ''); ?><br /><span class="text-base font-thin text-black">All Rights Reserved</span>
            </p>
            
            <div class="mt-4 pt-2 mx-auto max-w-sm">
                <p class="font-thin text-sm text-black flex items-center justify-center gap-1.5">
                    Powered by <i data-lucide="shield-check" class="w-4 h-4 text-green-700"></i>
                    <span class="font-black text-[var(--color-secondary)] tracking-wide">
                        <?php echo htmlspecialchars($branding['name'] ?? ''); ?><span class="font-black text-black tracking-wide"><?php echo htmlspecialchars($branding['crm'] ?? ''); ?></span>™
                    </span>
                </p>
            </div>
        </div>
       </div>
       
    </div>
  </main>

  <aside class="hidden md:flex w-1/2 relative">
    <img src="<?php echo htmlspecialchars($branding['login-bg']); ?>" alt="Fondo Login" class="absolute inset-0 w-full h-full object-cover">
    <div class="absolute inset-0 bg-gradient-to-tr from-[var(--color-primary)]/70 via-[var(--color-primary)]/40 to-[var(--color-secondary)]/40"></div>

    <div class="relative z-10 flex flex-col justify-between w-full p-10">
      <div class="flex items-center gap-3">
        <img src="<?php echo htmlspecialchars($branding['favicon']); ?>" class="h-9 w-9 rounded-lg" alt="Logo">
        <span class="text-white/90 font-semibold tracking-wide"><?php echo htmlspecialchars($branding['biz_name']); ?></span>
      </div>

      <div class="text-white">
        <div class="flex justify-start mb-6">
          <img src="<?php echo htmlspecialchars($branding['flow-logo']); ?>" alt="<?php echo htmlspecialchars($branding['full_title']); ?>" class="h-16 xl:h-20 w-auto">
        </div>

        <p class="mt-3 text-white/85 max-w-xl text-lg font-black">
          La Plataforma #1 <span class="text-[var(--color-highlight)]">Para Impulsar Tu Negocio</span>
        </p>

        <ul class="mt-6 grid grid-cols-1 lg:grid-cols-2 gap-3 text-white/90 max-w-2xl">
          <li class="flex items-center gap-3"><i data-lucide="users" class="w-5 h-5"></i> Gestión De Prospectos y Clientes</li>
          <li class="flex items-center gap-3"><i data-lucide="workflow" class="w-5 h-5"></i> Pipeline Visual y Seguimiento</li>
          <li class="flex items-center gap-3"><i data-lucide="calendar-clock" class="w-5 h-5"></i> Agenda, Citas y Recordatorios</li>
          <li class="flex items-center gap-3"><i data-lucide="file-text" class="w-5 h-5"></i> Cotizaciones y Finanzas</li>
        </ul>
      </div>
    </div>
  </aside>

  <script>
    lucide.createIcons();
    const toggleBtn = document.getElementById('togglePass');
    const passInput = document.getElementById('password');
    let visible = false;
    toggleBtn?.addEventListener('click', () => {
      visible = !visible;
      passInput.type = visible ? 'text' : 'password';
      toggleBtn.innerHTML = visible
        ? '<i data-lucide="eye-off" class="w-5 h-5"></i>'
        : '<i data-lucide="eye" class="w-5 h-5"></i>';
      lucide.createIcons();
      passInput.focus();
    });
  </script>
  <script src="files/toast.js"></script>
</body>
</html>