/**
 * portal_actions.js
 * Maneja la lógica de envío asíncrono (AJAX) para las acciones de Citas del Cliente.
 * Utiliza los scripts del CRM: agenda-create.php, agenda-update.php, agenda-delete.php.
 */

document.addEventListener('DOMContentLoaded', () => {
    const bookingForm = document.getElementById('booking-form');

    // --- Rutas a los scripts del CRM (Asumiendo que están en el directorio superior) ---
    const CREATE_URL = '../db/agenda-create.php'; 
    const UPDATE_URL = '../db/agenda-update.php';
    const DELETE_URL = '../db/agenda-delete.php';

    // Función para manejar la creación de citas (POST)
    if (bookingForm) {
        bookingForm.addEventListener('submit', async function(e) {
            e.preventDefault();

            const submitButton = bookingForm.querySelector('button[type="submit"]');
            const originalButtonText = submitButton.innerHTML;

            submitButton.disabled = true;
            submitButton.innerHTML = 'Sending Request... <i data-lucide="loader-2" class="w-5 h-5 ml-2 animate-spin"></i>';
            window.lucide.createIcons();

            const formData = new FormData(bookingForm);
            const data = {};
            formData.forEach((value, key) => data[key] = value);
            
            // CRÍTICO: Añadir una nota por defecto, ya que agenda-create.php lo requiere
            if (!data.notes) {
                data.notes = 'Appointment requested by client from web portal.';
            }

            try {
                const response = await fetch(CREATE_URL, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(data)
                });

                const result = await response.json();
                
                if (result.status === 'success') {
                    window.showNotification(result.message || 'Appointment successfully requested.', 'success'); 
                    setTimeout(() => window.location.reload(), 2000); 
                } else {
                    window.showNotification(result.message || 'Unknown error while scheduling.', 'error');
                }

            } catch (error) {
                console.error('Error al enviar la solicitud:', error);
                window.showNotification('Connection error with the scheduling system.', 'error');
            } finally {
                submitButton.disabled = false;
                submitButton.innerHTML = originalButtonText;
                window.lucide.createIcons();
            }
        });
    }

    /**
     * Maneja la actualización de ESTADO (Cancelar Cita) usando agenda-update.php
     * @param {string} citaIdFull - ID completo (ej. cita-unica-123)
     * @param {string} citaType - 'unica' o 'recurrente'
     * @param {string} newStatus - El nuevo estado a aplicar (ej. 'cancelled')
     */
    async function handleUpdateStatus(citaIdFull, citaType, newStatus) {
        // [CORRECCIÓN CRÍTICA] Diálogo de confirmación en INGLÉS
        if (!confirm(`Are you sure you want to mark this appointment as ${newStatus.toUpperCase()}?`)) return;

        // El script de tu CRM (agenda-update.php) espera el ID numérico
        const numericId = citaIdFull.split('-').pop();

        try {
            const response = await fetch(UPDATE_URL, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                // JSON esperado por agenda-update.php para actualización de estado
                body: JSON.stringify({ id: numericId, type: citaType, status: newStatus }) 
            });

            const result = await response.json();

            if (result.status === 'success') {
                window.showNotification(`Appointment successfully marked as ${newStatus}.`, 'success'); 
                setTimeout(() => window.location.reload(), 1500); 
            } else {
                window.showNotification(result.message || 'Error updating appointment status.', 'error');
            }
        } catch (error) {
            console.error('Connection error:', error);
            window.showNotification('Connection error with your CRM server.', 'error');
        }
    }

    /**
     * Maneja la ELIMINACIÓN de citas usando agenda-delete.php
     * @param {string} citaIdFull - ID completo (ej. cita-unica-123)
     * @param {string} citaType - 'unica' o 'recurrente'
     */
    async function handleDeleteAppointment(citaIdFull, citaType) {
        // [CORRECCIÓN CRÍTICA] Diálogo de confirmación en INGLÉS
        if (!confirm('Are you sure you want to permanently delete this appointment? This action cannot be undone.')) return;

        try {
            const response = await fetch(DELETE_URL, {
                method: 'DELETE',
                headers: { 'Content-Type': 'application/json' },
                // JSON esperado por agenda-delete.php
                body: JSON.stringify({ id: citaIdFull, type: citaType }) 
            });

            const result = await response.json();

            if (result.status === 'success') {
                window.showNotification('Appointment successfully deleted.', 'success'); 
                setTimeout(() => window.location.reload(), 1500); 
            } else {
                window.showNotification(result.message || 'Error deleting appointment.', 'error');
            }
        } catch (error) {
            console.error('Connection error:', error);
            window.showNotification('Connection error with your CRM server.', 'error');
        }
    }
    
    // --- Asignar Listeners a los botones de la lista de citas activas ---
    
    function setupActiveAppointmentListeners() {
        // 1. CANCELAR (UPDATE Status to 'cancelled')
        document.querySelectorAll('.action-cancel-btn').forEach(button => {
            button.onclick = function() {
                const id = this.dataset.citaId;
                const type = this.dataset.citaType;
                handleUpdateStatus(id, type, 'cancelled');
            };
        });
        
        // 2. ELIMINAR (DELETE)
        document.querySelectorAll('.action-delete-btn').forEach(button => {
            button.onclick = function() {
                const id = this.dataset.citaId;
                const type = this.dataset.citaType;
                handleDeleteAppointment(id, type);
            };
        });

        // 3. EDITAR (Simular)
        document.querySelectorAll('.edit-client-cita-btn').forEach(button => {
            button.onclick = function() {
                // En un entorno de portal de cliente, la edición es limitada. 
                window.showNotification('Please contact support to reschedule or modify a confirmed appointment.', 'info');
            };
        });
    }

    // Llamar a la función al cargar la página
    setupActiveAppointmentListeners();
    
    // Exportar la función para que se pueda usar si la lista se recarga dinámicamente
    window.setupActiveAppointmentListeners = setupActiveAppointmentListeners; 
});