<?php ob_start();
require_once '../files/guardiankey.php'; ?>
<?php
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../logs/php_errors.log');
error_reporting(E_ALL);

header('Content-Type: application/json');

// Incluye el archivo de conexión a la base de datos
require_once 'db_connection.php';

$response = ['status' => 'error', 'message' => 'Solicitud no válida.'];

if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
    $data = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Error al decodificar JSON en agenda-update.php: " . json_last_error_msg());
        echo json_encode(['status' => 'error', 'message' => 'Error al procesar los datos enviados.']);
        exit;
    }

    // ✅ CORRECCIÓN: Validar y sanitizar ID como entero
    $id = isset($data['id']) ? filter_var($data['id'], FILTER_VALIDATE_INT) : null;
    $type = filter_var($data['type'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $new_status = filter_var($data['status'] ?? null, FILTER_SANITIZE_FULL_SPECIAL_CHARS);

    // Validar ID y tipo
    if ($id === false || empty($type)) {
        $response['message'] = 'Faltan datos obligatorios (id, type) o son inválidos.';
        echo json_encode($response);
        exit;
    }

    try {
        // --- Manejar actualización de estado (si se envía el campo 'status') ---
        if ($new_status !== null) { // Usar !== null para asegurar que se intentó enviar un estado
            if ($type === 'unica') {
                $stmt = $pdo->prepare("UPDATE unique_appointments SET status = ? WHERE id = ?");
                $stmt->execute([$new_status, $id]);
                $response = ['status' => 'success', 'message' => 'Estado de cita única actualizado con éxito.'];
            } elseif ($type === 'recurrente') {
                // Las citas recurrentes no tienen un campo 'status' en la tabla.
                // Esta acción no se guarda en la base de datos para citas recurrentes.
                // Si la lógica de negocio requiere el seguimiento de instancias completadas/canceladas de citas recurrentes,
                // se necesitaría una tabla intermedia (ej. `recurring_appointment_instances`)
                // o un cambio en el esquema de `recurring_appointments` para un estado global.
                // Por ahora, se simula el éxito para que el frontend actualice su estado local.
                $response = ['status' => 'success', 'message' => 'Operación de estado para cita recurrente reconocida, pero no aplicable a la base de datos directamente.'];
            } else {
                $response['message'] = 'Tipo de cita no válido para actualización de estado.';
            }
            echo json_encode($response); // Envía respuesta y sale después de actualizar el estado (o simularlo)
            exit;
        }

        // --- Manejar actualización de otros campos (edición general) ---
        $notes = filter_var($data['notes'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        
        if ($type === 'unica') {
            $appointment_date = filter_var($data['appointment_date'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
            $appointment_time = filter_var($data['appointment_time'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

            if (empty($appointment_date) || empty($appointment_time)) {
                $response['message'] = 'Faltan datos obligatorios para actualizar cita única (appointment_date, appointment_time).';
                echo json_encode($response);
                exit;
            }

            $stmt = $pdo->prepare("UPDATE unique_appointments SET appointment_date = ?, appointment_time = ?, notes = ? WHERE id = ?");
            $stmt->execute([$appointment_date, $appointment_time, $notes, $id]);
            $response = ['status' => 'success', 'message' => 'Cita única actualizada con éxito.'];

        } elseif ($type === 'recurrente') {
            $frequency = filter_var($data['frequency'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

            if (empty($frequency)) {
                $response['message'] = 'Faltan datos obligatorios para actualizar cita recurrente (frequency).';
                echo json_encode($response);
                exit;
            }

            $stmt = $pdo->prepare("UPDATE recurring_appointments SET frequency = ?, notes = ? WHERE id = ?");
            $stmt->execute([$frequency, $notes, $id]);
            $response = ['status' => 'success', 'message' => 'Cita recurrente actualizada con éxito.'];

        } else {
            $response['message'] = 'Tipo de cita no reconocido para actualización de campos.';
        }

    } catch (PDOException $e) {
        error_log("Error PDO al actualizar cita en agenda-update.php: " . $e->getMessage());
        $response['message'] = 'Error de base de datos al actualizar cita. Por favor, inténtalo de nuevo más tarde.';
    } catch (Exception $e) {
        error_log("Error general al actualizar cita en agenda-update.php: " . $e->getMessage());
        $response['message'] = 'Error interno del servidor al actualizar cita. Por favor, inténtalo de nuevo más tarde.';
    }
}

ob_end_clean();
echo json_encode($response);