<?php
ob_start(); // Inicia el búfer de salida

// Carga la configuración y limpia cualquier salida que pueda haber generado
require_once __DIR__ . '/../config.php';
ob_clean();

header('Content-Type: application/json; charset=utf-8');

try {
    // 1. VERIFICACIÓN DE API KEY
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada o está vacía en config.php.']);
        exit;
    }
    if (!defined('GEMINI_MODEL') || empty(GEMINI_MODEL)) {
        define('GEMINI_MODEL', 'gemini-2.5-flash'); // Fallback si no está definida
    }
    $apiKey = GEMINI_API_KEY;

    // 2. VERIFICACIÓN DEL MÉTODO HTTP
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido. Solo se aceptan solicitudes POST.']);
        exit;
    }

    // 3. OBTENCIÓN Y DECODIFICACIÓN DE DATOS DE ENTRADA
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido en la solicitud.']);
        exit;
    }

    // =========================================================================
    // Captura y Sanitización de Contexto e Historial
    // =========================================================================
    $context = [
        // Capturamos el nuevo campo value_proposition
        'value_proposition' => trim($data['context']['value_proposition'] ?? ''), 
        'client_name' => trim($data['context']['client_name'] ?? ''),
        'service_info' => trim($data['context']['service_info'] ?? ''), // NUEVO CAMPO AÑADIDO
        'ad_copy' => trim($data['context']['ad_copy'] ?? ''),
        'ideal_client_summary' => trim($data['context']['ideal_client_summary'] ?? '')
    ];

    $history = [];
    if (isset($data['history']) && is_array($data['history'])) {
        foreach ($data['history'] as $msg) {
            $history[] = [
                'role' => trim($msg['role'] ?? 'unknown'),
                'text' => trim($msg['text'] ?? '')
            ];
        }
    }

    // Validar contexto obligatorio
    if (empty($context['client_name']) || empty($context['service_info']) || empty($context['ideal_client_summary']) || empty($history)) {
        http_response_code(400);
        echo json_encode(['error' => 'Faltan datos de contexto o historial de conversación obligatorios.']);
        exit;
    }
    
    // Construir la variable $historyString y obtener el último mensaje
    $historyString = "";
    foreach ($history as $message) {
        // Usamos la lógica de mapeo 'user' y 'model' a 'Cliente' y 'Vendedor'
        $roleLabel = ($message['role'] === 'user') ? "Cliente" : "Vendedor";
        // Si el rol es 'model', lo tratamos como una respuesta del vendedor anterior (para mantener el hilo)
        $historyString .= "{$roleLabel}: \"{$message['text']}\"\n";
    }

    $currentClientMessage = end($history)['text'];
    
    // El último mensaje del cliente es el que debemos analizar, por lo que lo quitamos del historial
    // NOTA: La lógica de `end($history)` ya lo obtiene, pero el frontend envía el mensaje actual en el último elemento.
    // Usaremos el último elemento del array $history como el mensaje a responder, si el rol es 'user'.
    
    // 5. CONSTRUCCIÓN DEL PROMPT PARA LA IA
    $prompt = "Actúa como un Director de Estrategia de Ventas B2C, un experto en cerrar tratos utilizando principios de neuroventas, manejo de objeciones y persuasión ética. Tu única misión es darle al vendedor la respuesta exacta que necesita para guiar al prospecto hacia el siguiente paso lógico de la venta.

    ## CONTEXTO CLAVE
    - **Nombre del Prospecto:** \"{$context['client_name']}\"
    - **Propuesta De Valor:** \"{$context['value_proposition']}\" 
    - **Servicio Principal:** \"{$context['service_info']}\"
    - **Anuncio Original (Su primer interés):** \"{$context['ad_copy']}\"
    - **Perfil del Cliente Ideal (Sus posibles dolores y deseos):** \"{$context['ideal_client_summary']}\"
    - **Historial de la Conversación:**
    {$historyString}
    - **Último Mensaje del Cliente (El desafío actual):**
    \"{$currentClientMessage}\"
    
    ## TU TAREA ESTRATÉGICA
    Analiza el **contexto completo** y genera **exactamente 2 sugerencias de respuesta**. Cada sugerencia debe emplear una táctica psicológica de ventas distinta y comprobada. No des respuestas genéricas; adáptalas a la perfección a la situación actual.
    
    ## ESTRUCTURA DE SALIDA (JSON Obligatorio)
    
    Para cada una de las 2 sugerencias, entrega:
    
    1.  **`titulo_estrategia`**: El nombre de la táctica de ventas utilizada. Ejemplos: 'Técnica del Espejo + Anclaje de Valor', 'Reencuadre de Objeción con Pregunta Pivote', 'Prueba Social y Reducción de Riesgo'.
    2.  **`explicacion`**: Una explicación de 15 a 25 palabras sobre por qué esta táctica funciona en este caso específico. **Enfócate en el resultado práctico**, no en la teoría.
    3.  **`respuesta`**: Un objeto con la respuesta para el cliente.
        * `es`: El mensaje en español.
        * `en`: La traducción precisa al inglés.
    
    ## REGLAS DE ORO PARA LAS RESPUESTAS
    -   **Cero Jerga Técnica:** Traduce cualquier característica en un **beneficio tangible y emocionante**. No digas 'CRM', di 'un sistema para que nunca más se te olvide un seguimiento'. No digas 'SEO', di 'aparecer primero en Google cuando te busquen'.
    -   **Lenguaje Humano y Directo:** Escribe como un humano real hablaría por WhatsApp. Usa un tono amigable, seguro y profesional.
    -   **Siempre Personalizado:** Integra el nombre del cliente (`{$context['client_name']}`) de forma natural.
    -   **Avanza la Venta:** Cada respuesta debe terminar sutilmente con una pregunta o una afirmación que invite al siguiente paso. Nunca dejes la conversación en un punto muerto.
    -   **Valores Reales, no Placeholders:** No uses `[Nombre]` o `{{producto}}` en la respuesta final.
    -   **Formato Limpio:** Usa `\\n` para párrafos claros y fáciles de leer.
    ";
    
    // 6. PREPARACIÓN Y EJECUCIÓN DE LA LLAMADA A LA API
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . $apiKey;
    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'sugerencias' => [
                        'type' => 'ARRAY',
                        'items' => [
                            'type' => 'OBJECT',
                            'properties' => [
                                'titulo_estrategia' => ['type' => 'STRING'],
                                'explicacion' => ['type' => 'STRING'],
                                'respuesta' => [
                                    'type' => 'OBJECT',
                                    'properties' => [
                                        'es' => ['type' => 'STRING'],
                                        'en' => ['type' => 'STRING']
                                    ], 'required' => ['es', 'en']
                                ]
                            ], 'required' => ['titulo_estrategia', 'explicacion', 'respuesta']
                        ]
                    ]
                ], 'required' => ['sugerencias']
            ]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    // 7. MANEJO DE ERRORES DE LA LLAMADA A LA API
    if ($response_ai === false || $http_code !== 200) {
        http_response_code(500);
        $error_details = $curl_error ?: ($response_ai ? json_decode($response_ai, true) : 'Respuesta de la API vacía o ilegible.');
        // Log al archivo correcto
        error_log("Error de conexión/respuesta de IA en api-sales-assistant.php: HTTP Code " . $http_code . ", cURL Error: " . $curl_error . ", Raw: " . json_encode($error_details, JSON_UNESCAPED_UNICODE));
        echo json_encode(['error' => 'No se pudo obtener una respuesta de la IA (Código HTTP: ' . $http_code . ').','details' => $error_details], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // 8. PROCESAMIENTO Y VALIDACIÓN DE LA RESPUESTA DE LA IA
    $result = json_decode($response_ai, true);
    $generated_text = $result['candidates'][0]['content']['parts'][0]['text'] ?? null;

    if ($generated_text === null) {
        http_response_code(500);
        error_log("La IA devolvió un formato de respuesta inesperado en api-sales-assistant.php.");
        echo json_encode(['error' => 'La IA devolvió un formato de respuesta inesperado.', 'details' => 'No se encontró la clave de texto generada.'], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $cleaned_text = str_replace(['```json', '```'], '', $generated_text);
    $final_data = json_decode(trim($cleaned_text), true);

    if ($final_data === null || !isset($final_data['sugerencias'])) {
        http_response_code(500);
        error_log("La IA devolvió un formato de JSON inválido o no contiene las sugerencias esperadas en api-sales-assistant.php.");
        echo json_encode(['error' => 'La IA devolvió un formato de JSON inválido o no contiene las sugerencias esperadas.', 'details' => json_last_error_msg()], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // 9. ENVÍO DE LA RESPUESTA FINAL
    ob_end_clean();
    echo json_encode($final_data, JSON_UNESCAPED_UNICODE);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    error_log("Excepción interna en api-sales-assistant.php: " . $e->getMessage());
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()], JSON_UNESCAPED_UNICODE);
    exit;
} finally {
    if (ob_get_level() > 0) {
        ob_end_clean();
    }
}
?>