<?php
// /IAMarketing/api/api-post-title-generator.php
ob_start();
require_once __DIR__ . '/../config.php';
ob_clean();

header('Content-Type: application/json');

try {
    // 1. VERIFICACIÓN DE SEGURIDAD Y DATOS
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido. Solo se aceptan solicitudes POST.']);
        exit;
    }

    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada.']);
        exit;
    }
    
    $apiKey = GEMINI_API_KEY;
    $data = json_decode(file_get_contents('php://input'), true);

    // Validamos los campos que vienen del formulario
    if (json_last_error() !== JSON_ERROR_NONE || !isset($data['service']) || !isset($data['ideal_client_summary']) || !isset($data['value_proposition'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Faltan campos requeridos (service, ideal_client_summary, value_proposition). ' . json_last_error_msg()]);
        exit;
    }

// 2. PROMPT MAESTRO
$prompt = "Actúa como un **copywriter de respuesta directa** especializado en crear 'hooks' (ganchos) para imágenes de anuncios en redes sociales.

--- REGLA PRINCIPAL ---
El texto debe ser **CORTO, IMPACTANTE y DIRECTO**. Ideal para superponer sobre una imagen. Máximo 6-8 palabras.

--- CONTEXTO DEL NEGOCIO ---
- **Servicio Principal:** {$data['service']}
- **Cliente Ideal (Problema/Deseo):** {$data['ideal_client_summary']}
- **Propuesta de Valor:** {$data['value_proposition']}

--- TU TAREA ---
Genera **3 OPCIONES DE TÍTULOS** para una imagen de anuncio. Cada opción debe tener un enfoque psicológico distinto y venir con su traducción al inglés.

1.  **Opción 1 (Enfoque: EL PROBLEMA):** Ataca directamente el dolor o la frustración del cliente.
2.  **Opción 2 (Enfoque: LA SOLUCIÓN/BENEFICIO):** Enfócate en el resultado final o el sentimiento deseado.
3.  **Opción 3 (Enfoque: EL SERVICIO/OFERTA):** Menciona el servicio o una oferta de forma clara y directa.

--- FORMATO DE RESPUESTA REQUERIDO (JSON VÁLIDO) ---
Genera un objeto JSON con 3 claves principales: `problema`, `solucion`, `servicio`.
Cada clave debe ser un objeto con dos subclaves: `es` (español) y `en` (inglés).

**Ejemplo de estructura de respuesta:**
{
  \"problema\": {
    \"es\": \"¿Cansado del desorden en casa?\",
    \"en\": \"Tired of home clutter?\"
  },
  \"solucion\": {
    \"es\": \"Tu Hogar en Completa Paz\",
    \"en\": \"Your Home in Complete Peace\"
  },
  \"servicio\": {
    \"es\": \"Limpieza Profunda Profesional\",
    \"en\": \"Professional Deep Cleaning\"
  }
}
";

    // 3. LLAMADA A LA API CON ESQUEMA FORZADO
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . $apiKey;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'problema' => [
                        'type' => 'OBJECT',
                        'properties' => [
                            'es' => ['type' => 'STRING'],
                            'en' => ['type' => 'STRING']
                        ],
                        'required' => ['es', 'en']
                    ],
                    'solucion' => [
                        'type' => 'OBJECT',
                        'properties' => [
                            'es' => ['type' => 'STRING'],
                            'en' => ['type' => 'STRING']
                        ],
                        'required' => ['es', 'en']
                    ],
                    'servicio' => [
                        'type' => 'OBJECT',
                        'properties' => [
                            'es' => ['type' => 'STRING'],
                            'en' => ['type' => 'STRING']
                        ],
                        'required' => ['es', 'en']
                    ]
                ],
                'required' => ['problema', 'solucion', 'servicio']
            ]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);

    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    // 4. PROCESAMIENTO Y ENVÍO DE LA RESPUESTA (CON DEPURACIÓN)
    if ($response_ai === false || $http_code !== 200) {
        http_response_code(502); // Bad Gateway
        echo json_encode([
            'error' => 'No se pudo obtener una respuesta válida del servicio de IA.',
            'curl_error' => $curl_error,
            'api_http_code' => $http_code,
            'api_response_body' => json_decode($response_ai)
        ]);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    $response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? null;

    if ($response_text === null) {
        http_response_code(500);
        echo json_encode(['error' => 'La respuesta de la IA no contenía el formato esperado (partes de texto).', 'raw_response' => $response_ai]);
        exit;
    }

    $response_data = json_decode($response_text, true);

    if ($response_data === null || !isset($response_data['problema'])) {
        http_response_code(500);
        echo json_encode(['error' => 'La IA devolvió una respuesta con formato JSON inválido.', 'raw_ai_text' => $response_text]);
        exit;
    }

    ob_end_clean();
    echo json_encode($response_data);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    if (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>