<?php
// /IAMarketing/api/api-offers-promotions.php
ob_start();
require_once __DIR__ . '/../files/guardiankey.php';
require_once __DIR__ . '/../config.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada.']);
        exit;
    }
    
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Extracción y sanitización de datos (ejemplo simplificado)
    $businessName = htmlspecialchars($data['businessName'] ?? 'un negocio local B2C', ENT_QUOTES);
    $businessContact = htmlspecialchars($data['businessContact'] ?? 'en su sitio web', ENT_QUOTES);
    $businessWeb = htmlspecialchars($data['businessWeb'] ?? '', ENT_QUOTES);
    $offerObjective = htmlspecialchars($data['offerObjective'] ?? 'atraer nuevos clientes', ENT_QUOTES);
    $mainService = htmlspecialchars($data['mainService'] ?? 'un servicio esencial', ENT_QUOTES);
    $discountType = htmlspecialchars($data['discountType'] ?? 'un descuento', ENT_QUOTES);
    $discountValue = htmlspecialchars($data['discountValue'] ?? 'una gran ventaja', ENT_QUOTES);
    $valueProposition = htmlspecialchars($data['value_proposition'] ?? 'proporciona resultados rápidos y confiables.', ENT_QUOTES);
    
    // Mapeo de objetivo a texto más legible (para el prompt)
    $objectiveMap = [
        'referrals' => 'generar referidos',
        'neighborhood' => 'captar clientes en un vecindario específico',
        'marketplace' => 'destacar en un marketplace o plataforma de servicios',
        'service_bundle' => 'impulsar un servicio o paquete específico',
        'seasonal' => 'aprovechar una temporada o festividad'
    ];
    $objectiveText = $objectiveMap[$offerObjective] ?? $offerObjective;
    
    // Mapeo de tipo de descuento a texto más legible (para el prompt)
    $discountText = match($discountType) {
        'percentage' => "un porcentaje del $discountValue%",
        'amount' => "un descuento de $$discountValue",
        'free_service' => "un servicio gratuito: $discountValue",
        default => $discountValue
    };

    $prompt = "
Actúa como un estratega de marketing directo y copywriter persuasivo, experto en crear ofertas y promociones irresistibles para negocios de servicios locales B2C. Tu objetivo es generar promociones que no solo ofrezcan un descuento, sino que generen una urgencia y un beneficio emocional claro para el cliente.

**Contexto del Negocio:**
- **Nombre:** {$businessName}
- **Propuesta de Valor Principal:** '{$valueProposition}'
- **Contacto:** {$businessContact}
- **Web/Red:** {$businessWeb}
- **Servicio a Promocionar:** {$mainService}
- **Objetivo de la Campaña:** {$objectiveText}
- **Tipo de Incentivo:** Ofrece {$discountText}.

**Tu Tarea:**
Genera DOS OPCIONES de Oferta/Promoción. Cada opción debe ser radicalmente distinta en su enfoque (Ej: Una enfocada en escasez/urgencia, y la otra en valor/transformación).
Cada opción debe ser un objeto con CUATRO campos: `offer_name`, `offer_description`, `promo_text_es`, y `promo_text_en`.

**Estructura de la Promoción (a incluir en promo_text_es y promo_text_en):**
1.  Un TÍTULO que atrape la atención y mencione la escasez/beneficio.
2.  Un CUERPO PERSUASIVO que explique el problema que resuelven y el valor del descuento.
3.  Un LLAMADO A LA ACCIÓN CLARO y la forma de contacto.
4.  Debes incluir el nombre del negocio y la URL/Contacto en el texto promocional.

**Reglas Importantes:**
- **NO uses un lenguaje corporativo o robótico.** Sé directo, humano y enérgico.
- **La respuesta DEBE ser un único objeto JSON válido** con una clave principal llamada `offers` que contenga un array de 2 objetos.
- **Asegúrate de que los campos de texto (`promo_text_es` y `promo_text_en`) usen saltos de línea `\n` para formateo visual.**

**Formato de Salida Requerido:**
La respuesta DEBE ser un único objeto JSON válido con la siguiente estructura:

```json
{
  \"offers\": [
    {
      \"offer_name\": \"[Nombre Corto y Atractivo de la Oferta 1]\",
      \"offer_description\": \"[Breve explicación del enfoque de marketing de esta opción]\",
      \"promo_text_es\": \"[Texto completo y persuasivo en español con saltos de línea]\",
      \"promo_text_en\": \"[Traducción fiel y persuasiva en inglés con saltos de línea]\"
    },
    {
      \"offer_name\": \"[Nombre Corto y Atractivo de la Oferta 2]\",
      \"offer_description\": \"[Breve explicación del enfoque de marketing de esta opción]\",
      \"promo_text_es\": \"[Texto completo y persuasivo en español con saltos de línea]\",
      \"promo_text_en\": \"[Traducción fiel y persuasiva en inglés con saltos de línea]\"
    }
  ]
}
";

$apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash:generateContent?key=' . GEMINI_API_KEY;

// Definición del esquema para validar y asegurar el formato de la respuesta
$responseSchema = [
    'type' => 'OBJECT',
    'properties' => [
        'offers' => [
            'type' => 'ARRAY',
            'description' => 'Array con las dos opciones de ofertas generadas.',
            'items' => [
                'type' => 'OBJECT',
                'properties' => [
                    'offer_name' => ['type' => 'STRING', 'description' => 'Nombre corto y atractivo de la oferta.'],
                    'offer_description' => ['type' => 'STRING', 'description' => 'Explicación del enfoque de la oferta.'],
                    'promo_text_es' => ['type' => 'STRING', 'description' => 'Texto promocional completo en español.'],
                    'promo_text_en' => ['type' => 'STRING', 'description' => 'Texto promocional completo en inglés.']
                ],
                'required' => ['offer_name', 'offer_description', 'promo_text_es', 'promo_text_en']
            ]
        ]
    ],
    'required' => ['offers']
];

$payload = [
    'contents' => [['parts' => [['text' => $prompt]]]],
    'generationConfig' => [
        'response_mime_type' => 'application/json',
        'response_schema' => $responseSchema
    ]
];

$ch = curl_init($apiUrl);
curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true, CURLOPT_POSTFIELDS => json_encode($payload), CURLOPT_HTTPHEADER => ['Content-Type: application/json'], CURLOPT_TIMEOUT => 90]);
$response_ai = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($response_ai === false || $http_code !== 200) {
    http_response_code(502);
    // Intentar decodificar si hay un error reportado por la API de Google, sino dar error genérico.
    $error_message = 'No se pudo obtener una respuesta válida del servicio de IA.';
    if ($response_ai !== false) {
         $error_data = json_decode($response_ai, true);
         $error_message .= " Detalles: " . ($error_data['error']['message'] ?? 'Error desconocido');
    } else {
         $error_message .= " Detalles: Error cURL.";
    }
    
    echo json_encode(['error' => $error_message]);
    exit;
}

$result_array = json_decode($response_ai, true);

$raw_response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? '';
$response_data = json_decode($raw_response_text, true);

if ($response_data === null || !isset($response_data['offers']) || count($response_data['offers']) !== 2) {
    http_response_code(500);
    echo json_encode(['error' => 'La IA no devolvió un array de 2 ofertas con las claves esperadas.']);
    exit;
}

ob_end_clean();
echo json_encode($response_data);
exit;
} ?>


### api-db/ia-offers-promotions-create.php

```php
<?php
// /IAMarketing/api-db/ia-offers-promotions-create.php
ob_start();
require_once __DIR__ . '/../client-information.php'; 
ob_clean();

header('Content-Type: application/json');

if (!isset($pdo)) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database connection error.']);
    exit;
}

$data = json_decode(file_get_contents('php://input'), true);

// 1. Validar campos requeridos
if (
    !isset($data['offer_name'], $data['offer_description'], $data['promo_text_es'], $data['promo_text_en'], $data['inputs']['mainService'], $data['inputs']['offerObjective'])
) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Datos incompletos para crear la oferta.']);
    exit;
}

// 2. Asignar datos. business_info_id es 1 por defecto en tu arquitectura.
$business_id = 1; 
$offer_name = $data['offer_name'];
$offer_description = $data['offer_description'];
$promo_text_es = $data['promo_text_es'];
$promo_text_en = $data['promo_text_en'];
$service_input = $data['inputs']['mainService'];
$objective_input = $data['inputs']['offerObjective'];

try {
    // Verificar si ya se ha alcanzado el límite de 5 ofertas (opcional pero recomendado)
    $check_limit_sql = "SELECT COUNT(id) AS offer_count FROM ia_offers_promotions_results WHERE business_info_id = ?";
    $check_limit_stmt = $pdo->prepare($check_limit_sql);
    $check_limit_stmt->execute([$business_id]);
    $row = $check_limit_stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($row && $row['offer_count'] >= 5) {
        http_response_code(409); // Conflicto: Límite Alcanzado
        echo json_encode(['success' => false, 'message' => 'Límite de 5 ofertas guardadas alcanzado. Por favor, elimina una para guardar esta.']);
        exit;
    }
    
    // 3. Insertar el nuevo registro en la tabla dedicada
    $sql = "INSERT INTO ia_offers_promotions_results 
            (business_info_id, offer_name, offer_description, promo_text_es, promo_text_en, service_input, objective_input) 
            VALUES (?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        $business_id, 
        $offer_name, 
        $offer_description, 
        $promo_text_es, 
        $promo_text_en,
        $service_input,
        $objective_input
    ]);
    
    echo json_encode(['success' => true, 'message' => 'Oferta creada y guardada con éxito.']);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database error while creating offer: ' . $e->getMessage()]);
}
?>