<?php
// /IAMarketing/api/api-planificador-de-objetivos.php
ob_start();
require_once __DIR__ . '/../config.php'; // Incluye config.php para GEMINI_API_KEY y GEMINI_MODEL
ob_clean(); // Limpia cualquier salida de búfer accidental

header('Content-Type: application/json');

try {
    // 1. VERIFICACIÓN DE SEGURIDAD Y DATOS
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido. Solo se aceptan solicitudes POST.']);
        exit;
    }

    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada en el servidor. Verifica tu archivo config.php.']);
        exit;
    }

    $apiKey = GEMINI_API_KEY; // Asigna la clave de API desde config.php

    $data = json_decode(file_get_contents('php://input'), true);

    // Validar campos requeridos del formulario
    $required_form_keys = ['main_goal', 'short_term_goal', 'medium_term_goal', 'long_term_goal', 'prior_experience', 'campaign_objective', 'has_website', 'other_strategies', 'budget', 'service_price', 'value_proposition'];
    foreach ($required_form_keys as $key) {
        if (!isset($data[$key])) {
            http_response_code(400);
            echo json_encode(['error' => "Faltan datos en la solicitud. Campo requerido: '$key'. " . json_last_error_msg()]);
            exit;
        }
    }

    $budget = floatval($data['budget']);
    $service_price = floatval($data['service_price']);
    $breakeven_clients = ($service_price > 0) ? ceil($budget / $service_price) : 0;
    $value_proposition = $data['value_proposition']; // Contexto de la arquitectura

    // 2. PROMPT MAESTRO (v1.9 - Con Propuesta de Valor)
    $prompt = "Actúa como un coach de negocios y estratega de marketing para dueños de negocios locales. Tu objetivo es analizar la situación completa de un estudiante para darle un feedback claro, realista y un plan de acción 100% aplicable, hablando siempre en segunda persona (Tú, Tus metas...).

--- FILOSOFÍA DEL CURSO ---
El curso CLIENT3 enseña a crear un sistema predecible. Se recomienda empezar con campañas de 'Recibir Mensajes' y un presupuesto mínimo de \$10 al día (\$300 al mes) para obtener datos fiables.

--- TU TAREA ESPECÍFICA ---
Un estudiante ha definido su situación. Analiza sus datos a través del lente de la FILOSOFÍA DEL CURSO y genera un reporte estratégico.

--- DATOS DEL ESTUDIANTE ---
- Propuesta de Valor (Contexto Clave): {$value_proposition}
- Objetivo Principal: {$data['main_goal']}
- Objetivo a Corto Plazo: {$data['short_term_goal']}
- Objetivo a Mediano Plazo: {$data['medium_term_goal']}
- Objetivo a Largo Plazo: {$data['long_term_goal']}
- Experiencia Previa: {$data['prior_experience']}
- Objetivo Para Tu Negocio: {$data['campaign_objective']}
- ¿Tiene Página Web?: {$data['has_website']}
- Otras Estrategias: {$data['other_strategies']}
- Presupuesto Mensual: \${$budget}
- Precio del Servicio: \${$service_price}

--- CÁLCULOS REALIZADOS ---
- Clientes para Cubrir Inversión: {$breakeven_clients}

--- ANÁLISIS CRÍTICO Y PERSONALIZADO REQUERIDO ---
1.  **Habla Directamente al Estudiante:** Usa siempre la segunda persona ('Tus objetivos...', 'Tu presupuesto...', 'Tu plan...').
2.  **Analiza la Realidad del Presupuesto:** Si el presupuesto mensual es menor a \$300, DEBES señalar que está por debajo del mínimo recomendado de \$10 al día y que esto hará que los resultados sean muy lentos o nulos. Sé directo pero constructivo.
3.  **Analiza la Realidad del Objetivo vs. Presupuesto:** Compara la meta de clientes del 'Objetivo Principal' con el 'Presupuesto'. Si es matemáticamente irrealista (ej. conseguir 100 clientes con \$100), DEBES explicar por qué es inviable de forma clara (ej. 'Tu meta de conseguir 100 clientes con un presupuesto de \$100 significa que tendrías que conseguir cada cliente por solo \$1, lo cual es extremadamente difícil en el mercado actual...').
4.  **Basa el Plan en la Experiencia y Estrategia:** El plan de acción debe ser 100% coherente con todos los datos proporcionados.
5.  **Coherencia de Objetivo de Negocio:** Asegúrate de que el 'Objetivo Para Tu Negocio' se refleje en el análisis y el plan de acción.

--- FORMATO DE RESPUESTA REQUERIDO ---
Genera un objeto JSON con las siguientes claves:
- 'analisis_objetivos': Un párrafo analizando la claridad de TUS metas.
- 'analisis_viabilidad': Un párrafo explicando el análisis financiero, incluyendo el punto de equilibrio y el análisis crítico de viabilidad.
- 'plan_de_accion': Un objeto con 'primer_paso', 'corto_plazo', 'mediano_plazo', y 'largo_plazo', dando un plan de acción coherente y personalizado.
- 'metricas_clave': Un párrafo con las métricas más importantes que DEBES monitorear.
- 'recomendaciones_optimizacion': Un párrafo con recomendaciones para optimizar campañas."
    ;

    // 3. LLAMADA A LA API CON ESQUEMA FORZADO Y MODELO CENTRALIZADO
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . $apiKey;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'analisis_objetivos' => ['type' => 'STRING', 'description' => 'Un párrafo analizando la claridad de las metas del estudiante.'],
                    'analisis_viabilidad' => ['type' => 'STRING', 'description' => 'Un párrafo explicando el análisis financiero y la viabilidad.'],
                    'plan_de_accion' => [
                        'type' => 'OBJECT',
                        'properties' => [
                            'primer_paso' => ['type' => 'STRING', 'description' => 'El primer paso inmediato del plan de acción.'],
                            'corto_plazo' => ['type' => 'STRING', 'description' => 'El enfoque a corto plazo (esta semana).'],
                            'mediano_plazo' => ['type' => 'STRING', 'description' => 'El enfoque a mediano plazo (este mes).'],
                            'largo_plazo' => ['type' => 'STRING', 'description' => 'El enfoque a largo plazo (3 meses).']
                        ],
                        'required' => ['primer_paso', 'corto_plazo', 'mediano_plazo', 'largo_plazo']
                    ],
                    'metricas_clave' => ['type' => 'STRING', 'description' => 'Un párrafo con las métricas clave a monitorear.'],
                    'recomendaciones_optimizacion' => ['type' => 'STRING', 'description' => 'Un párrafo con recomendaciones para la optimización.']
                ],
                'required' => ['analisis_objetivos', 'analisis_viabilidad', 'plan_de_accion', 'metricas_clave', 'recomendaciones_optimizacion']
            ]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);

    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch); // Capturar errores de cURL
    curl_close($ch);

    // 4. PROCESAMIENTO Y ENVÍO DE LA RESPUESTA (CON DEPURACIÓN)
    if ($response_ai === false || $http_code !== 200) {
        http_response_code(502); // Bad Gateway
        echo json_encode([
            'error' => 'No se pudo obtener una respuesta válida del servicio de IA.',
            'curl_error' => $curl_error,
            'api_http_code' => $http_code,
            'api_response_body' => json_decode($response_ai)
        ]);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    $response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? null;

    if ($response_text === null) {
        http_response_code(500);
        echo json_encode(['error' => 'La respuesta de la IA no contenía el formato esperado (partes de texto).', 'raw_response' => $response_ai]);
        exit;
    }

    $response_data = json_decode($response_text, true);

    if ($response_data === null) {
        http_response_code(500);
        echo json_encode(['error' => 'La IA devolvió una respuesta con formato JSON inválido.', 'raw_ai_text' => $response_text, 'json_last_error' => json_last_error_msg()]);
        exit;
    }

    // Asegurarse de que todas las claves requeridas por el schema están presentes
    $required_keys = ['analisis_objetivos', 'analisis_viabilidad', 'plan_de_accion', 'metricas_clave', 'recomendaciones_optimizacion'];
    foreach ($required_keys as $key) {
        if (!isset($response_data[$key])) {
            http_response_code(500);
            echo json_encode(['error' => "La IA no devolvió la clave esperada: '$key'.", 'partial_data' => $response_data]);
            exit;
        }
    }

    // Si todo es correcto, enviar la respuesta validada.
    ob_end_clean();
    echo json_encode($response_data);
    exit;

} catch (Exception $e) {
    // Capturar cualquier excepción inesperada y devolver un JSON de error.
    http_response_code(500);
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    // Asegurarse de que el búfer de salida se detenga y limpie al final.
    if (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>