<?php
// /IAMarketing/api/api-local-market-analsyis.php
ob_start();
require_once __DIR__ . '/../config.php';
ob_clean();

header('Content-Type: application/json');

try {
    // Verificación de API Key
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada.']);
        exit;
    }
    $apiKey = GEMINI_API_KEY;

    // Verificación del método HTTP
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido.']);
        exit;
    }

    // Obtención y validación de datos
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido.']);
        exit;
    }

    // Sanitizar y validar las entradas
    $company_name = filter_var($data['company_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $value_proposition = filter_var($data['value_proposition'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS); // <-- CAMPO DE ARQUITECTURA
    $service_type = filter_var($data['service_type'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $location = filter_var($data['location'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $target_audience = filter_var($data['target_audience'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $price_range = filter_var($data['price_range'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $additional_details = filter_var($data['additional_details'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

    // Validar campos obligatorios
    if (empty($company_name) || empty($service_type) || empty($location)) {
        http_response_code(400);
        echo json_encode(['error' => 'Faltan campos obligatorios: Nombre, Tipo de Servicio y Ubicación.']);
        exit;
    }

    // Construcción del PROMPT MAESTRO (Actualizado con value_proposition)
    $prompt = "Actúa como un experto en análisis de mercado para negocios de servicios locales. Tu objetivo es generar un reporte de mercado conciso y un plan estratégico accionable.

    **CONTEXTO DEL NEGOCIO:**
    - Nombre del Negocio: \"{$company_name}\"
    - Propuesta de Valor (Clave): \"{$value_proposition}\"
    - Tipo de Servicio: \"{$service_type}\"
    - Ubicación Principal: \"{$location}\"
    " . (!empty($target_audience) ? "- Público Objetivo Principal: \"{$target_audience}\"\n" : "") . "
    " . (!empty($price_range) ? "- Rango de Precios: \"{$price_range}\"\n" : "") . "
    " . (!empty($additional_details) ? "- Detalles Adicionales: \"{$additional_details}\"\n" : "") . "

    **TU TAREA:**
    Genera un análisis y un plan estratégico en 3 partes, redactado en un lenguaje 100% simple y sin jerga técnica para el dueño de un negocio.

    1.  **analisis_mercado:** Un párrafo de 3 a 4 oraciones sobre el estado del mercado local en la ubicación proporcionada para el servicio especificado.
    2.  **analisis_competencia:** Un párrafo de 3 a 4 oraciones que resuma quiénes son los posibles competidores y qué oportunidades existen para que este negocio destaque (basado en su propuesta de valor).
    3.  **plan_estrategico:** Un arreglo de 3 a 5 puntos que contengan acciones concretas y prácticas para diferenciarse y atraer más clientes. Cada punto debe ser una frase clara y accionable.

    **REGLAS IMPORTANTES:**
    -   **NO uses marcadores de posición** como `[Nombre del Negocio]` en el resultado final.
    -   **NO incluyas texto introductorio o explicaciones** fuera del objeto JSON.
    -   La salida debe ser un **objeto JSON válido** con las claves \"analisis_mercado\", \"analisis_competencia\" y \"plan_estrategico\" (este último debe ser un array).
    ";

    // --- ARQUITECTURA MEJORADA ---
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . $apiKey;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'analisis_mercado' => ['type' => 'STRING'],
                    'analisis_competencia' => ['type' => 'STRING'],
                    'plan_estrategico' => ['type' => 'ARRAY', 'items' => ['type' => 'STRING']]
                ],
                'required' => ['analisis_mercado', 'analisis_competencia', 'plan_estrategico']
            ]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);

    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    // --- MANEJO DE ERRORES DEPURABLE ---
    if ($response_ai === false || $http_code !== 200) {
        http_response_code(502); // Bad Gateway
        error_log("Error de conexión/respuesta de IA: HTTP Code " . $http_code . ", cURL Error: " . $curl_error . ", Raw: " . $response_ai);
        echo json_encode([
            'error' => 'No se pudo obtener una respuesta válida del servicio de IA.',
            'curl_error' => $curl_error,
            'api_http_code' => $http_code,
            'api_response_body' => json_decode($response_ai)
        ]);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    $response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? null;
    
    if ($response_text === null) {
        http_response_code(500);
        error_log("La respuesta de la IA no contenía el formato esperado. Raw: " . json_encode($result_array));
        echo json_encode(['error' => 'La respuesta de la IA no contenía el formato esperado.']);
        exit;
    }

    // El archivo original tenía esta limpieza, la respetamos.
    $response_text = str_replace(['```json', '```'], '', $response_text);
    $final_data = json_decode(trim($response_text), true);

    if ($final_data === null || !isset($final_data['analisis_mercado']) || !isset($final_data['analisis_competencia']) || !isset($final_data['plan_estrategico'])) {
        http_response_code(500);
        error_log("La IA devolvió un formato de JSON inválido. Raw AI Text: " . trim($response_text));
        echo json_encode(['error' => 'La IA devolvió un formato de JSON inválido.']);
        exit;
    }

    // Envío de la respuesta final
    ob_end_clean();
    echo json_encode($final_data, JSON_UNESCAPED_UNICODE);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    error_log("Excepción interna: " . $e->getMessage());
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    if (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>