<?php
// /api/api-gmb-profile-optimizer.php
ob_start();
require_once __DIR__ . '/../config.php';

header('Content-Type: application/json');

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido. Solo se aceptan solicitudes POST.']);
        exit;
    }

    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada en el servidor.']);
        exit;
    }

    $apiKey = GEMINI_API_KEY;
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido.']);
        exit;
    }

    if (!isset($data['pending_tasks']) || !is_array($data['pending_tasks']) || empty($data['pending_tasks'])) {
        http_response_code(400);
        echo json_encode(['error' => 'No se recibieron tareas pendientes para generar la estrategia.']);
        exit;
    }

    $pending_tasks_string = implode("\n", $data['pending_tasks']);

    // PROMPT MEJORADO PARA DEVOLVER JSON
    $prompt = "
    Actúa como un experto en SEO Local, especializado en optimizar Perfiles de Negocio de Google (Google My Business) para PYMES.
    Un dueño de negocio ha identificado los siguientes puntos débiles:
    {$pending_tasks_string}

    TU MISIÓN:
    Basado en esos puntos, crea un Plan de Acción Táctico en JSON.

    REGLAS DE FORMATO JSON (OBLIGATORIAS):
    - El JSON debe ser un objeto.
    - Debe tener una clave 'main_title' (string, ej: 'Tu Plan de Acción para un Perfil de Google Exitoso').
    - Debe tener una clave 'intro_text' (string, un párrafo corto y motivador).
    - Debe tener una clave 'action_points' (array de objetos).
    - Cada objeto en 'action_points' debe tener:
      - 'id': Un ID único en formato 'step-X' (ej: 'step-1', 'step-2').
      - 'title': El TÍTULO de la acción (string, ej: 'Configura tus Categorías Correctamente').
      - 'why': El PORQUÉ es importante (string, ej: 'Es la señal más fuerte para...').
      - 'steps': Un array de strings con los pasos a seguir (ej: ['Ve a tu Perfil...', 'Haz clic en...']).
    - Asegúrate de que los títulos sean claros y los pasos sean accionables.
    ";

    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . GEMINI_API_KEY;
    
    // DEFINICIÓN DEL ESQUEMA JSON DE SALIDA
    $schema = [
        'type' => 'OBJECT',
        'properties' => [
            'main_title' => ['type' => 'STRING'],
            'intro_text' => ['type' => 'STRING'],
            'action_points' => [
                'type' => 'ARRAY',
                'items' => [
                    'type' => 'OBJECT',
                    'properties' => [
                        'id' => ['type' => 'STRING'],
                        'title' => ['type' => 'STRING'],
                        'why' => ['type' => 'STRING'],
                        'steps' => ['type' => 'ARRAY', 'items' => ['type' => 'STRING']]
                    ],
                    'required' => ['id', 'title', 'why', 'steps']
                ]
            ]
        ],
        'required' => ['main_title', 'intro_text', 'action_points']
    ];

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json', // Forzar salida JSON
            'response_schema' => $schema,             // Aplicar el esquema
            'temperature' => 0.7,
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);
    
    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($response_ai === false || $http_code !== 200) {
        http_response_code(502);
        error_log("Error de IA en API (GMB): HTTP " . $http_code . ", cURL: " . $curl_error . ", Raw Response: " . $response_ai);
        echo json_encode(['error' => 'No se pudo obtener una respuesta válida del servicio de IA. Inténtalo de nuevo.']);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    
    if (!isset($result_array['candidates'][0]['content']['parts'][0]['text'])) {
        http_response_code(500);
        error_log("La respuesta de la IA (GMB) no tuvo el formato esperado. Raw: " . $response_ai);
        echo json_encode(['error' => 'La respuesta de la IA llegó con un formato inesperado.']);
        exit;
    }
    
    // La respuesta de la IA ya es un string JSON, lo decodificamos para enviarlo como JSON real
    $response_json_text = $result_array['candidates'][0]['content']['parts'][0]['text'];
    $response_data = json_decode($response_json_text, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(500);
        error_log("La IA devolvió un JSON inválido. Raw: " . $response_json_text);
        echo json_encode(['error' => 'La IA devolvió un JSON inválido.']);
        exit;
    }

    ob_end_clean();
    // Devolvemos el objeto JSON decodificado, que será recodificado por json_encode
    echo json_encode($response_data);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    error_log("Excepción en API (GMB): " . $e.getMessage());
    echo json_encode(['error' => 'Error interno del servidor: ' . $e.getMessage()]);
    exit;
}
?>