<?php
// /IAMarketing/api/api-business-diagnostic-radar.php
header('Content-Type: application/json');
ob_start();

require_once __DIR__ . '/../config.php'; 

try {
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La API Key de Gemini no está configurada.']);
        exit;
    }

    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido.']);
        exit;
    }

    $data = json_decode(file_get_contents('php://input'), true);

    if (!$data || (!isset($data['situations']) || empty($data['situations'])) && (!isset($data['business_description']) || empty($data['business_description']))) {
        http_response_code(400);
        echo json_encode(['error' => 'No se seleccionó ninguna situación ni se proporcionó una descripción.']);
        exit;
    }

    $situations_list = !empty($data['situations']) ? "- " . implode("\n- ", $data['situations']) : "No se seleccionaron problemas específicos.";
    $business_description = $data['business_description'] ?? 'No proporcionada.';
    // --- CONTEXTO ADICIONAL DE LA ARQUITECTURA ---
    $value_proposition = $data['value_proposition'] ?? 'No definida.';

    // --- PROMPT MAESTRO (v5.1 - Con Propuesta de Valor) ---
    $prompt = "
    Actúa como un **consultor de negocios senior y mentor del 'Programa Empresarial ECO'**. Tu especialidad es realizar diagnósticos precisos basados en un modelo de 8 ejes estratégicos y crear planes de acción estrictamente priorizados. Tu lógica debe ser infalible.

    --- FILOSOFÍA Y 8 EJES ESTRATÉGICOS (ORDEN DE PRIORIDAD ABSOLUTO) ---
    El crecimiento de un negocio se construye en 3 fases, siguiendo 8 ejes en orden:
    **FASE 1: FUNDACIÓN (Las bases)**
    1.  **Objetivos y Estrategia:** El pilar maestro. Sin esto, nada tiene sentido.
    2.  **Identidad Corporativa:** La imagen profesional que genera confianza inicial.
    3.  **Presencia Digital:** Canales (web, redes) que funcionan para atraer y comunicar valor.
    **FASE 2: ESTRUCTURACIÓN (Los procesos)**
    4.  **Canales de Comunicación:** Un sistema organizado para atender a los clientes.
    5.  **Procesos de Conversión:** Un embudo claro para guiar a los interesados hacia la venta.
    6.  **Estrategia Publicitaria:** Uso inteligente de anuncios pagados para generar un flujo predecible de clientes.
    **FASE 3: OPTIMIZACIÓN (El crecimiento)**
    7.  **Seguimiento Post-Venta:** Sistema para fidelizar, obtener reseñas y referidos.
    8.  **Monitoreo y Evaluación:** Medir resultados para tomar decisiones basadas en datos.

    --- TU TAREA ESPECÍFICA ---
    Un emprendedor ha completado el Radar de Diagnóstico. Analiza sus respuestas:

    **Descripción del Negocio:**
    {$business_description}

    **Propuesta de Valor del Negocio (Contexto Clave):**
    {$value_proposition}

    **Problemas Específicos Seleccionados:**
    {$situations_list}

    Realiza 3 cosas con una lógica estricta:
    1.  **Asigna una 'Puntuación de Madurez Digital' de 1 a 100 con REGLAS DE CASTIGO:**
        - Comienza con 100 puntos. Por cada problema seleccionado, resta puntos.
        - **REGLA CRÍTICA DE CASTIGO:** La puntuación final está limitada por el eje más bajo con problemas.
            - Si hay CUALQUIER problema en el **Eje 1**, la puntuación MÁXIMA posible es **40**.
            - Si no hay problemas en el Eje 1, pero sí en el **Eje 2**, la puntuación MÁXIMA posible es **55**.
            - Si no hay problemas en 1-2, pero sí en el **Eje 3**, la puntuación MÁXIMA posible es **65**.
        - Los problemas en ejes bajos (1-3) restan más puntos (8-10 pts). Los de ejes medios (4-6) restan moderadamente (5-7 pts). Los de ejes altos (7-8) restan menos (3-5 pts).

    2.  **Escribe un 'Diagnóstico General':**
        - Primero, determina la fase del negocio ('Etapa de Fundación', 'Etapa de Estructuración' o 'Etapa de Optimización') basado en el eje más bajo con problemas.
        - Escribe un párrafo que comience nombrando esa etapa. Luego, resume por qué se encuentra ahí, mencionando las áreas más débiles de forma directa y constructiva.

    3.  **Crea un 'Plan de Acción Personalizado':**
        - Un array de strings.
        - **REGLA 1:** El plan debe contener **un paso de acción por cada eje estratégico que tenga problemas**. Si el usuario seleccionó problemas en 5 ejes diferentes, el array debe contener 5 strings.
        - **REGLA 2:** Los pasos deben estar **ESTRICTAMENTE PRIORIZADOS** según el orden de los 8 ejes. El primer paso DEBE abordar el problema del eje con el número más bajo.
        - **REGLA 3:** Para cada paso, indica **QUÉ** debe hacer y **POR QUÉ** es importante, conectándolo con el eje estratégico. Resalta conceptos clave con negritas (`**palabra**`).

    --- FORMATO DE RESPUESTA REQUERIDO ---
    La respuesta DEBE ser un único objeto JSON válido.
    **REGLA FINAL Y MÁS IMPORTANTE:** CADA PASO DEL `plan_de_accion` DEBE SER UN STRING INDEPENDIENTE DENTRO DEL ARRAY. NO COMBINES MÚLTIPLES PASOS NUMERADOS DENTRO DE UN SOLO STRING.
    ";

    // --- ARQUITECTURA MEJORADA ---
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . GEMINI_API_KEY;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'puntuacion' => ['type' => 'NUMBER', 'description' => 'Puntuación numérica de 1 a 100.'],
                    'diagnostico_general' => ['type' => 'STRING', 'description' => 'Un párrafo resumiendo el estado actual del negocio.'],
                    'plan_de_accion' => [
                        'type' => 'ARRAY',
                        'description' => 'Un array de strings con los pasos a seguir. Cada paso es un string separado.',
                        'items' => ['type' => 'STRING']
                    ]
                ],
                'required' => ['puntuacion', 'diagnostico_general', 'plan_de_accion']
            ]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);

    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch); // Capturamos el error de cURL
    curl_close($ch);

    // --- MANEJO DE ERRORES DEPURABLE ---
    if ($response_ai === false || $http_code !== 200) {
        http_response_code(502); // Bad Gateway
        echo json_encode([
            'error' => 'No se pudo obtener una respuesta válida del servicio de IA.',
            'curl_error' => $curl_error,
            'api_http_code' => $http_code,
            'api_response_body' => json_decode($response_ai)
        ]);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    $raw_response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? '';
    $response_data = json_decode($raw_response_text, true);

    if ($response_data === null || !isset($response_data['puntuacion'])) {
        http_response_code(500);
        echo json_encode([
            'error' => 'La IA no devolvió un diagnóstico con el formato esperado.',
            'raw_response' => $raw_response_text
        ]);
        exit;
    }

    ob_end_clean();
    echo json_encode($response_data);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    while (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>